//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.1
import QtQuick.Controls 1.1 as QuickControls
import QtQuick.Controls.Styles 1.1

import QtGraphicalEffects 1.0

import wesual.Controls 1.0

FocusScope {
    id : comboBox

    property var presets : [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 ]

    property int value : 0
    property int minimum  : 0
    property int maximum  : 10

    readonly property bool open : popup.visible

    onValueChanged : {
        if (value < minimum)
            value = minimum;
        if (value > maximum)
            value = maximum;
        popup.currentSelection = presets.indexOf(value);
    }

    onMaximumChanged : {
        if (minimum < maximum)
            minimum = maximum;
        if (value > maximum)
            value = maximum;
    }

    onMinimumChanged : {
        if (maximum < minimum)
            maximum = minimum;
        if (value < minimum)
            value = minimum;
    }

    function openPopup()  {
        popup.show();
    }

    Component {
        id : presetItem

        ComboBoxItem {}
    }

    Component.onCompleted : {
        try {
            var items = [];
            for (var i = 0; i < presets.length; ++i) {
                if (typeof presets[i] !== "number") {
                    throw new Error("Presets must be integer values");
                }
                var item = presetItem.createObject(popup, {
                    "text" : presets[i].toFixed(0)
                });
                items.push(item);
            }
            popup.items = items;
            comboBox.value = presets[0];
        } catch (e) {
            console.error(e.message);
        }
    }

    // Key handling
    Keys.onSpacePressed : {
        if (!popup.visible) {
            comboBox.openPopup();
        }
    }
    Keys.onUpPressed : {
        popup.currentSelection = Math.max(0, popup.currentSelection - 1);
        value = presets[popup.currentSelection];
    }
    Keys.onDownPressed : {
        popup.currentSelection =
                       Math.min(presets.length - 1, popup.currentSelection + 1);
        value = presets[popup.currentSelection];
    }
    Keys.onRightPressed : {}
    Keys.onLeftPressed : {}
    Keys.onPressed : {
        if (event.key === Qt.Key_PageUp || event.key === Qt.Key_PageDown) {
            // do not propagate navigation keys while focus item
            event.accepted = true;
        }
    }


    implicitWidth  : 54
    implicitHeight : 26
    z : popup.visible ? 10000 : 0
    activeFocusOnTab : true

    Rectangle {
        id : background

        z : 1
        anchors.fill : parent
        color : UI.color(UI.PrimaryBase)
        border {
            width : 1
            color : {
                if (mouseArea.pressed)
                    return UI.color(UI.ControlBorderPress);
                else if (mouseArea.containsMouse || comboBox.activeFocus)
                    return UI.color(UI.ControlBorderHover);
                else
                    return UI.color(UI.PrimaryControlBorder);
            }
        }

        Behavior on border.color {
            ColorAnimation { duration : 0 }
        }

        QuickControls.TextField {
            anchors {
                fill : parent
                leftMargin : 0
                rightMargin : indicatorWrapper.width
            }
            text : comboBox.value.toFixed(0)
            style : TextFieldStyle {
                background : Item {}

                font {
                    family : UI.fontFamily(UI.PrimaryFont)
                    weight : UI.fontWeight(UI.PrimaryFont)
                    pixelSize : 14
                }

                textColor : UI.color(UI.PrimaryTextColor)
                selectedTextColor : UI.color(UI.PrimaryTextColor)
                selectionColor : UI.color(UI.TextSelection)
            }
            verticalAlignment : Qt.AlignVCenter
            validator : IntValidator {
                bottom : comboBox.minimum
                top : comboBox.maximum
            }
            onAccepted : {
                comboBox.value = parseInt(text);
                focus = false;
                popup.open = false;
            }
            onActiveFocusChanged : {
                if (!activeFocus && acceptableInput) {
                    comboBox.value = parseInt(text);
                    popup.open = false;
                }
            }
        }

        Rectangle {
            id : indicatorWrapper
            width : 23
            height : parent.height - 2
            anchors {
                right : parent.right
                rightMargin : 1
                verticalCenter : parent.verticalCenter
            }
            color : {
                if (mouseArea.pressed || popup.open)
                    return UI.color(UI.PrimaryPress);
                else if (mouseArea.containsMouse)
                    return UI.color(UI.PrimaryHover);
                else
                    return UI.color(UI.PrimaryBase);
            }
            Behavior on color {
                ColorAnimation { duration : 0 }
            }

            Image {
                id : indicator
                anchors {
                    right : parent.right
                    rightMargin : 6
                    verticalCenter : parent.verticalCenter
                    verticalCenterOffset : 1
                }
                source : "qrc:/controls/icons/dropdown-indicator.png"
                visible : false
            }
            ColorOverlay {
                anchors.fill : indicator
                source : indicator
                color : mouseArea.pressed || popup.open ?
                        UI.color(UI.PrimaryBase) :
                        UI.color(UI.DropdownIndicatorIcon)
            }
        }
    }

    MouseArea {
        id : mouseArea

        z : 2
        hoverEnabled : true
        width : indicatorWrapper.width
        anchors {
            top : parent.top
            right : parent.right
            bottom : parent.bottom
        }
        acceptedButtons : Qt.LeftButton
        onPressed : comboBox.openPopup()
    }

    ComboBoxPopup {
        id : popup
        z : 0
        items : comboBox.items
        comboBox : comboBox

        onOpenChanged : {
            if (open) {
                focus = true;
                activeIndex = popup.currentSelection;
            } else {
                focus = false;
            }
        }

        onAccepted : {
            try {
                comboBox.value = presets[result];
            } catch (e) {
                console.error(e.message);
            }
        }

        function updatePresets() {
            try {
                var items = [];
                for (var i = 0; i < presets.length; ++i) {
                    if (typeof presets[i] !== "number") {
                        throw new Error("Presets must be integer values");
                    }
                    var item = presetItem.createObject(popup, {
                        "text" : presets[i].toFixed(0)
                    });
                    items.push(item);
                }
                popup.items = items;
            } catch (e) {
                console.error(e.message);
            }
        }
    }
}

